package loquebot.util;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.NavPoint;

/**
 * Contain some useful static routines..
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public abstract class LoqueUtils
{

    /**
     * Approx-compares given values, using given measure of ignorance.
     * Randomizes, when the two values are, within range of ignorance, the same.
     * @param a First value.
     * @param b Second value.
     * @param ignorance Amount of ignorance.
     * @return True, if a is less than b. False otherwise.
     */
    public static boolean approxCompare (int a, int b, int ignorance)
    {
        if (a + ignorance < b)
            return true;
        if (b + ignorance < a)
            return false;
        return (Math.random () < .5);
    }

    /*========================================================================*/

    /**
     * Chooses from given locations the one, which is nearest to the start.
     * Note: The chosen location is removed from the given list and returned.
     * @param start Starting location.
     * @param locations List of locations to choose from.
     * @param ignorance Amount of ignorance. See {@link #approxCompare}.
     * @return Chosen location.
     */
    public static Triple pickNearestLocation (Triple start, ArrayList<Triple> locations, int ignorance)
    {
        // no locations to choose from?
        if (locations.size () <= 0)
            return null;

        // choose the first one first
        int chosenKey = 0;
        int chosenDst = (int) Triple.distanceInSpace (start, locations.get (0));

        // run through the rest of the locations
        for (int i = locations.size () - 1; i > 0; i--)
        {
            // get the distance..
            int currentDst = (int) Triple.distanceInSpace (start, locations.get (i));

            // if this location is closer
            if (approxCompare (currentDst, chosenDst, ignorance))
            {
                // choose this new one..
                chosenKey = i;
                chosenDst = currentDst;
            }
        }

        // return the chosen key and return it
        return locations.remove(chosenKey);
    }

    /*========================================================================*/

    /**
     * Calculates comparable distance between the two given places and along
     * the given path.
     *
     * <h4>Pogamut troubles</h4>
     *
     * This method would also like to include additional info from navigation
     * points. For example, there may be additional expense set in navigation
     * points.
     *
     * <p>For now, this method at least adds constant value for each mover.</p>
     *
     * @param start Starting location.
     * @param path Path to follow.
     * @param dest Destination location.
     * @return Distance from start to destination.
     */
    public static int distanceAlongPath (Triple start, ArrayList<NavPoint> path, Triple dest)
    {
        int distance = 0;

        // calculate distances along the path...
        for (NavPoint np : path)
        {
            // distance from the previous node to this one
            distance += (int)Triple.distanceInSpace (start, np.location);
            // add special distance for movers
            if (np.UnrealID.indexOf ("LiftCenter") > -1)
                distance += 400;
            // move the start along
            start = np.location;
        }

        // and now, there is just one node left.. the final destination
        return distance + (int)Triple.distanceInSpace (start, dest);
    }




}